/* Copyright (C) 2012-2018 RealVNC Ltd. All Rights Reserved.
 */

#ifndef __VNCDEVICEPROVIDERCALLBACKS_H__
#define __VNCDEVICEPROVIDERCALLBACKS_H__

#include <vncdiscoverysdktypes.h>
#include <stddef.h>
#include <vncdiscoverysdkutils.h>

/**
 * \file vncdeviceprovidercallbacks.h
 *
 * This file defined the Device Provider Callback API. The Device Provider uses
 * this to communicate back to the Discovery SDK.
 *
 * This file should not be included directly. vncdeviceprovider.h should be used
 * instead.
 *
 * \see VNCDeviceProviderCallbacks
 * \see vncdeviceprovider.h
 */

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * \brief The typedef for the VNCDeviceProvider. The implementation should be a
 * structure with the name VNCDeviceProviderImpl.
 */
typedef struct VNCDeviceProviderImpl VNCDeviceProvider;

/**
 * \brief Holds information about an interface of a device.
 */
typedef struct VNCDeviceInterface VNCDeviceInterface;

/**
 * \brief Holds information about an interface of a device.
 *
 * The device interface can have any meaning defined by the type of device
 * provider. It is recommended that each Device Provider documents the
 * interfaces and the properties that it can support for its devices.
 *
 * The structure must be allocated by using
 * VNCDeviceProviderAllocDeviceInterface and freed by using
 * VNCDeviceProviderFreeDeviceInterface.
 *
 * \see VNCDeviceInfo, VNCDeviceProviderAllocDeviceInterface,
 * VNCDeviceProviderFreeDeviceInterface
 */
struct VNCDeviceInterface
{
  /**
   * \brief The size of the interface structure.
   * 
   * This describes the size in bytes of the structure. It allows the structure
   * to be extended in the future without breaking compatibility.
   */
  vnc_uint32_t size;
  /**
   * \brief Value describing the type of interface.
   *
   * This value can be matched against the types of interfaces defined by
   * Device Provider.
   */
  vnc_int32_t type;
  /**
   * \brief Boolean value specifying if the interface is available or not.
   */
  vnc_int32_t available;

  /**
   * \brief The number of properties defined.
   */
  vnc_uint32_t propertiesCount;
  /**
   * \brief Array of property values.
   *
   * The property array contains pointers to string values. The meaning of each
   * property must be defined by the Device Provider. For example by having an
   * enumeration of property fields. If a property is NULL it means it is not
   * known/defined.
   *
   * The properties strings and the array itself is owned by the structure and
   * is freed when calling VNCDeviceProviderFreeDeviceInterface.
   *
   * \note The array is not NULL terminated. The propertyCount value specifies
   * how many properties are defined (although some properties can be NULL).
   */
  char **ppProperties;

  /**
   * \brief The number of defined subInterfaces.
   */
  vnc_uint32_t subInterfacesCount;
  /**
   * \brief Array of sub-interfaces.
   *
   * An interface can have several sub-interfaces. They are defined here.
   *
   * The array and all the subinterfaces are owned by this structure and are
   * free when calling VNCDeviceProviderFreeDeviceInterface.
   */
  VNCDeviceInterface **ppSubInterfaces;
};

/**
 * \brief Holds information about device and its interfaces.
 *
 * The values, properties, interfaces held by the device have different meaning
 * depending on the type of device. The Device Provider should describe the
 * types of devices it provides and the meaning of the fields of the types.
 *
 * The structure must be allocated by using
 * VNCDeviceProviderAllocDevice and freed by using
 * VNCDeviceProviderFreeDevice.
 *
 * \see VNCDeviceProviderCallbacks
 * \see VNCDeviceInterface, VNCDeviceProviderAllocDevice,
 * VNCDeviceProviderFreeDevice
 */
typedef struct
{
  /**
   * \brief The size of the device structure.
   * 
   * This describes the size in bytes of the structure. It allows the structure
   * to be extended in the future without breaking compatibility.
   */
  vnc_uint32_t size;
  /**
   * \brief The ID of the device.
   *
   * This is used to identify the device, for example when the device is
   * reported as removed. It must be unique.
   */
  vnc_uint32_t id;
  /**
   * \brief The address of the device.
   *
   * The address can be for example the path to the device, or the MAC address
   * of the device.
   *
   * This can be NULL if the type of device does not have a value.
   */
  char *pAddress;
  /**
   * \brief The name of the provider that found the device.
   *
   * This is a NUL termiated string. It is owned by this structure.
   */
  char *pProviderName;
  /**
   * \brief The type of device.
   *
   * This defines the type of device. Depending on the type, the meaning of all
   * the properties and interfaces can be defined. The type is device provider
   * dependent.
   */
  vnc_uint32_t type;

  /**
   * \brief The number of properties defined.
   */
  vnc_uint32_t propertiesCount;

  /**
   * \brief Array of property values.
   *
   * The property array contains pointers to string values. The meaning of each
   * property must be defined by the Device Provider. For example by having an
   * enumeration of property fields. If a property is NULL it means it is not
   * known/defined.
   *
   * The properties strings and the array itself is owned by the structure and
   * is freed when calling VNCDeviceProviderFreeDevice.
   *
   * \note The array is not NULL terminated. The propertyCount value specifies
   * how many properties are defined (although some properties can be NULL).
   */
  char **ppProperties;

  /**
   * \brief The number of defined interfaces.
   */
  vnc_uint32_t interfacesCount;
  /**
   * \brief Array of the interfaces of the device.
   *
   * An interface can have several interfaces. They are defined here.
   *
   * The array and all the interfaces are owned by this structure and are
   * free when calling VNCDeviceProviderFreeDevice.
   */
  VNCDeviceInterface **ppInterfaces;
} VNCDeviceInfo;

/**
 * \brief Notifies when the device provider has started.
 *
 * Informs the Discovery SDK whether the start operation on the Device Provider
 * instance was successful or not.
 *
 * The Device Provider has to notify that it is started before making any 
 * other notifications, or sending any other responses.
 *
 * If you want to set any timers you can do so after this call returns. If you
 * are restarting the Device Provider, make sure to restart any timers that you
 * need.
 *
 * \param pContext The context of the SDK for the DeviceProvider.
 * \param error VNCDiscoverySDKErrorNone if the start operation was succesful,
 *              or otherwise another error code. If the error code is different
 *              from VNCDiscoverySDKErrorNone then the SDK will not monitor the
 *              Device Provider handles.
 *
 * \see VNCDeviceProviderCallbacks
 * \see VNCDeviceProviderStart, VNCDiscoverySDKError
 */
typedef void VNCCALL
VNCDeviceProviderNotifyStarted(void *pContext,
    VNCDiscoverySDKError error);

/**
 * \brief Notifies when the device provider has stopped.
 *
 * Informs the Discovery SDK whether the stop operation on the Device Provider
 * instance was successful or not.
 *
 * Until this function is called, the Device Provider should not change the
 * Event Handler value. After this call, the SDK will not monitor the Event
 * Handler anymore and the Device Provider can modify it.
 *
 * After this call the SDK will not monitor the Event Handles of any of the
 * Devices. Also it will cancel all the timers.
 *
 * The Device Provider should not make any notification calls, or responses,
 * after it notified it's stopped.
 *
 * \param pContext The context of the SDK for the DeviceProvider.
 * \param error VNCDiscoverySDKErrorNone if the stop operation was succesful,
 *              or otherwise another error code.
 *
 * \see VNCDeviceProviderCallbacks
 * \see VNCDeviceProviderStop, VNCDiscoverySDKError
 */
typedef void VNCCALL
VNCDeviceProviderNotifyStopped(void *pContext,
    VNCDiscoverySDKError error);

/**
 * \brief Asks the SDK to set a timer for a handle of the Device Provider.
 *
 * This call asks the SDK to call back the Device Provider, for a specific
 * handle, after a certain time period. The handle must be known to the SDK
 * (i.e. must be present in the handles returned by
 * VNCDeviceProviderGetMultipleEventHandles).
 *
 * The SDK will only call back once, so for further notifications this methods
 * should be called again.
 *
 * The method should only be called when the Device Provider is started,
 * otherwise it will have no effect. Make sure the SDK is notified that the
 * Device Provider is started before making this call.
 *
 * To stop an existing timer, set the timeout value to VNCDiscoverySDKNoTimeout.
 *
 * To get an instant callback, set the timeout value to VNCDisocverySDKInstantCall.
 *
 * \param pContext The context of the SDK for the DeviceProvider.
 * \param pEventHandle The handle for which to set the timer. Should be
 * non-NULL and the value should be different from
 * VNCDeviceProviderNoEventHandle. The timeout will be set for the event handle
 * with the given value.
 * \param timeout The timeout value. It should not be negative. If you want to
 *            stop the timer set to VNCDiscoverySDKNoTimeout.
 *
 * \see VNCDeviceProviderCallbacks
 * \see VNCDeviceProviderGetEventHandlesEx, VNCDeviceProviderHandleTimeout
 */
typedef void VNCCALL
VNCDeviceProviderSetTimer(void *pContext,
    VNCDiscovererEventHandle *pEventHandle,
    VNCDiscoverySDKTimeoutMicroseconds timeout);

/**
 * \brief Notifies the SDK of updates to the event handle list.
 *
 * Notifies the Discovery SDK that the event handle list has changed. This
 * should be called to ensure that the SDK picks up the modified event handle
 * list when the Device Provider is using multiple event handles.
 *
 * It is not necessary to call this from within the SDK thread because the
 * event handle list will be re-checked by the SDK before waiting for an event
 * again.
 *
 * \param pContext The context of the SDK for the DeviceProvider.
 *
 * \see VNCDeviceProviderCallbacks
 * \see VNCDeviceProviderGetEventHandlesEx
 */
typedef void VNCCALL
VNCDeviceProviderEventHandlesChanged(void *pContext);

/**
 * \brief Notification that a device has been found.
 *
 * Informs the Discovery SDK that a device has been found. The device must not
 * have been reported to the SDK before. The Device Provider should report
 * devices that already existed when it was started.
 *
 * This method should only be called when the Device Provider is running (has
 * been started successfully).
 *
 * \param pContext The context of the SDK for the DeviceProvider.
 * \param pDevice Pointer to the found device. Ownership to the device is
 * transferred to the Discovery SDK. The device should be reported only once.
 *
 * \see VNCDeviceProviderCallbacks
 * \see VNCDeviceProviderStart, VNCDeviceProviderNotifyDeviceRemoved,
 * VNCDeviceInfo
 */
typedef void VNCCALL
VNCDeviceProviderNotifyDeviceFound(void *pContext,
    VNCDeviceInfo *pDevice);

/**
 * \brief Notification that an existing device is no longer available.
 *
 * Informs the Discovery SDK that an existing device is no longer available.
 *
 * This method should only be called when the Device Provider is running (has
 * been started successfully). Device Providers need to make sure to inform the
 * SDK if devices are removed.
 *
 * \param pContext The context of the SDK for the DeviceProvider.
 * \param deviceId The ID of the device that was removed.
 *
 * \see VNCDeviceProviderCallbacks
 * \see VNCDeviceProviderStart, VNCDeviceProviderNotifyDeviceFound,
 * VNCDeviceInfo, VNCDeviceProviderNotifyDeviceRemovedFromAddress
 */
typedef void VNCCALL
VNCDeviceProviderNotifyDeviceRemoved(void *pContext,
    vnc_uint32_t deviceId);

/**
 * \brief Notification that an existing device, at a specified address, is no
 * longer available.
 *
 * Informs the Discovery SDK that an existing device, from a given address, is
 * no longer available.
 *
 * This method should only be called when the Device Provider is running (has
 * been started successfully). Device Providers need to make sure to inform the
 * SDK if devices are removed.
 *
 * The advantage of this callback over VNCDeviceProviderNotifyDeviceRemoved is
 * that the Device Provider does not need to keep track of device IDs to see
 * which device is removed. The SDK may use the address to remove the correct
 * devices.
 *
 * \param pContext The context of the SDK for the DeviceProvider.
 * \param pAddress The address of the device that was removed.
 *
 * \see VNCDeviceProviderCallbacks
 * \see VNCDeviceProviderStart, VNCDeviceProviderNotifyDeviceFound,
 * VNCDeviceInfo, VNCDeviceProviderNotifyDeviceRemoved
 */
typedef void VNCCALL
VNCDeviceProviderNotifyDeviceRemovedFromAddress(void *pContext,
    const char *pAddress);

/**
 * \brief Logs a string for the Device Provider.
 *
 * This method is used by the Device Provider to send to the SDK logs.
 *
 * \param pContext The context of the SDK for the DeviceProvider.
 * \param severity The severity of the log.
 * \param pText The NUL-terminated string to log. This is owned by the Device
 * Provider so it should be freed after the log method returns.
 *
 * \see VNCDeviceProviderCallbacks
 * \see VNCDeviceProviderIsSeverityLogged, VNCDiscovererLogSeverity
 */
typedef void VNCCALL
VNCDeviceProviderLog(void *pContext,
    VNCDiscovererLogSeverity severity, 
    const char *pText);

/**
 * \brief Tells the Device Provider whether a certain severity will get logged.
 *
 * This method is used by the Device Provider to find out if a certain severity
 * will be logged to the client application. If the severity does not match the
 * client application options, then no logging would be performed. This method
 * can be used by the Device Provider to avoid composing unnescessary logs.
 *
 * \param pContext The context of the SDK for the DeviceProvider.
 * \param severity The severity of the potential log.
 *
 * \return 1 if the severity would get logged, 0 otherwise.
 */
typedef vnc_int32_t VNCCALL
VNCDeviceProviderIsSeverityLogged(void *pContext,
    VNCDiscovererLogSeverity severity);

/**
 * \brief Allocates memory for a new device.
 *
 * This call allocates enough memory for a new device. It also initialized the
 * arrays containing properties and interfaces. Both arrays will have all
 * elements initialized to NULL.
 *
 * \param propertiesCount The number of properties available for the device. The
 * array will be allocated and initialized to contain only NULL elements. Also
 * the propertyCount value will be set to this value.
 * \param interfacesCount The number of interfaces that the device has. The SDK
 * will allocate a big enough array of pointers to interfaces to contain as
 * many interfaces as defined here. All the pointers will be initialized to
 * NULL.
 *
 * \return A newly allocated device structure, or NULL if there is not enough
 * memory to provide it.
 *
 * \see VNCDeviceProviderCallbacks
 * \see VNCDeviceProviderFreeDevice, VNCDeviceProviderAllocDeviceInterface,
 * VNCDeviceInfo
 */
typedef VNCDeviceInfo *VNCCALL
VNCDeviceProviderAllocDevice(vnc_uint32_t propertiesCount,
    vnc_uint32_t interfacesCount);

/**
 * \brief Frees the memory allocated for a device.
 *
 * This call also frees any allocated structures held by the device, for
 * example the ID, address, interfaces and properties.
 *
 * \param pDeviceInfo The device to be freed.
 *
 * \see VNCDeviceProviderCallbacks
 * \see VNCDeviceProviderAllocDevice, VNCDeviceProviderAllocDeviceInterface,
 * VNCDeviceInfo
 */
typedef void VNCCALL
VNCDeviceProviderFreeDevice(VNCDeviceInfo *pDeviceInfo);

/**
 * \brief Allocates memory for a new device interface.
 *
 * This call allocates enough memory for a new device interface. It also
 * initialized the arrays containing properties and sub-interfaces. Both arrays
 * will have all elements initialized to NULL.
 *
 * \param propertiesCount The number of properties available for the interface.
 * The array will be allocated and initialized to contain only NULL elements.
 * Also the propertyCount value will be set to this value.
 * \param subInterfacesCount The number of sub-interfaces that the interface has.
 * The SDK will allocate a big enough array of pointers to sub-interfaces to
 * contain as many sub-interfaces as defined here. All the pointers will be
 * initialized to NULL.
 *
 * \return A newly allocated device interface structure, or NULL if there is
 * not enough memory to provide it.
 *
 * \see VNCDeviceProviderCallbacks
 * \see VNCDeviceProviderFreeDeviceInterface, VNCDeviceProviderAllocDevice,
 * VNCDeviceInterface
 */
typedef VNCDeviceInterface *VNCCALL
VNCDeviceProviderAllocDeviceInterface(vnc_uint32_t propertiesCount,
    vnc_uint32_t subInterfacesCount);

/**
 * \brief Frees the memory allocated for a device interface.
 *
 * This call also frees any allocated structures held by the device, for
 * example the interfaces and properties.
 *
 * If an interface is part of a device, calling VNCDeviceProviderFreeDevice
 * will free the interface as well, so there is no need to call this, unless
 * the interface does not belong to a device.
 *
 * \param pDeviceInfo The device to be freed.
 *
 * \see VNCDeviceProviderCallbacks
 * \see VNCDeviceProviderAllocDeviceInterface, VNCDeviceProviderAllocDevice,
 * VNCDeviceInterface
 */
typedef void VNCCALL
VNCDeviceProviderFreeDeviceInterface(VNCDeviceInterface *pDeviceInterface);

/**
 * \brief Allocates the memory for a string for the Device Provider.
 *
 * This method should be used if the Device Provider has to create a string and pass
 * it to the SDK. The method can allocate a new string that is a copy of the
 * original string, or can just allocate a certain amount of memory.
 * 
 * \param pString The original string. If the new string should be a copy of an
 *                original one this is where the original is specified. If the 
 *                new string is empty then this should be NULL
 * \param length The length of the string. If the first parameter is NULL then
 *               the length should be greater than 0. Otherwise, if there is an
 *               original string, the length specifies how much of the string
 *               to copy. Use -1 if you want the entire string copied.
 *
 * \return A pointer to the new string, or NULL if there is not enough memory
 *        (or NULL was passed initially together with -1 to allocString).
 */
typedef char *VNCCALL
VNCDeviceProviderAllocString(const char *pString, size_t length);

/**
 * \brief Frees the memory of a string allocated by the SDK.
 *
 * This is used to free any string that was allocated through the SDK.
 *
 * The provided string must be NUL terminated if non-NULL.
 *
 * \param pString The string to be freed.
 */
typedef void VNCCALL
VNCDeviceProviderFreeString(char *pString);

/**
 * \brief Allocates an array of key-value pairs.
 *
 * This allocates the array of key-value pairs.
 *
 * \param arraySize The size of the array to allocate.
 *
 * \return The array of key-value pairs.
 *
 * \see VNCDeviceProviderGetDeviceProperties
 */
typedef VNCDiscoverySDKKeyValPair *VNCCALL
VNCDeviceProviderAllocKeyValPairArray(size_t arraySize);

/**
 * \brief Frees a previously allocated array of key-value pairs.
 *
 * This can be used by the Device Provider if something fails when creating the
 * array of properties. The SDK will free the array, with all the elements and
 * the properties defined in them.
 *
 * \param pArray The array to free.
 * \param arraySize The size of the array.
 *
 * \see VNCDeviceProviderAllocKeyValPairArray
 */
typedef void VNCCALL
VNCDeviceProviderFreeKeyValPairArray(VNCDiscoverySDKKeyValPair *pArray,
    size_t arraySize);

/**
 * \brief Retrieves the utility functions supported by the SDK.
 *
 * This should be called only once the context has been set by the SDK (in
 * VNCDeviceProviderSetContext).
 *
 * \param pContext The context of the SDK for the DeviceProvider.
 * \param pUtils A pointer to a structure which will be filled with the utility
 * function.
 * \param utilsSize The size of the structure.
 *
 * \see vncdiscoverysdkutils.h
 * \see VNCDiscoverySDKUtils
 */
typedef void VNCCALL
VNCDeviceProviderGetUtils(void *pContext,
    VNCDiscoverySDKUtils *pUtils, size_t utilsSize);

/**
 * Returns a string explanation of the error code.
 *
 * \param error The error code to convert to string.
 *
 * \return The string representation of the error code. If the error code is
 * not known this information will be contained in the returned string.
 */
typedef const char *VNCCALL
VNCDeviceProviderErrorToString(VNCDiscoverySDKError error);

/**
 * \brief The Interface of the Callbacks API.
 *
 * The Device Provider should use this interface to notify the SDK,
 * to log, or allocate/free memory.
 *
 * \see VNCDeviceProviderInterface
 */
typedef struct
{
  /** Major component of the SDK's version number. */
  vnc_int32_t versionMajor;
  /** Minor component of the SDK's version number. */
  vnc_int32_t versionMinor;
  /** Patch component of the SDK's version number. */
  vnc_int32_t versionPatch;
  /** Build number component of the SDK's version number. */
  vnc_int32_t versionBuild;
  /**
   * Notifies when the device provider has started.
   */
  VNCDeviceProviderNotifyStarted *notifyStarted;
  /**
   * Notifies when the device provider has stopped.
   */
  VNCDeviceProviderNotifyStopped *notifyStopped;
  /**
   * Asks the SDK to set a timer for a handle of the Device Provider.
   */
  VNCDeviceProviderSetTimer *setTimer;
  /**
   * Notifies the SDK of updates to the event handle list.
   */
  VNCDeviceProviderEventHandlesChanged *eventHandlesChanged;
  /**
   * Notification that a device has been found.
   */
  VNCDeviceProviderNotifyDeviceFound *notifyDeviceFound;
  /**
   * Notification that an existing device is no longer available.
   */
  VNCDeviceProviderNotifyDeviceRemoved *notifyDeviceRemoved;
  /**
   * Notification that an existing device, at a specified address, is no
   * longer available.
   */
  VNCDeviceProviderNotifyDeviceRemovedFromAddress *notifyDeviceRemovedFromAddress;
  /**
   * Logs a string for the Device Provider.
   */
  VNCDeviceProviderLog *log;
  /**
   * Tells the Device Provider whether a certain severity will get logged.
   */
  VNCDeviceProviderIsSeverityLogged *isSeverityLogged;
  /**
   * Allocates memory for a new device.
   */
  VNCDeviceProviderAllocDevice *allocDevice;
  /**
   * Frees the memory allocated for a device.
   */
  VNCDeviceProviderFreeDevice *freeDevice;
  /**
   * Allocates memory for a new device interface.
   */
  VNCDeviceProviderAllocDeviceInterface *allocDeviceInterface;
  /**
   * Frees the memory allocated for a device interface.
   */
  VNCDeviceProviderFreeDeviceInterface *freeDeviceInterface;
  /**
   * Allocates the memory for a string for the Device Provider.
   */
  VNCDeviceProviderAllocString *allocString;
  /**
   * Frees the memory of a string allocated by the SDK.
   */
  VNCDeviceProviderFreeString *freeString;
  /**
   * Allocates an array of key-value pairs.
   */
  VNCDeviceProviderAllocKeyValPairArray *allocKeyValPairArray;
  /**
   * Frees a previously allocated array of key-value pairs.
   */
  VNCDeviceProviderFreeKeyValPairArray *freeArray;
  /**
   * Retrieves the utility functions supported by the SDK.
   */
  VNCDeviceProviderGetUtils *getUtils;
  /**
   * Returns a string explanation of the error code.
   */
  VNCDeviceProviderErrorToString *errorToString;
} VNCDeviceProviderCallbacks;

#ifdef __cplusplus
}
#endif

#endif /* !defined(__VNCDEVICEPROVIDERCALLBACKS_H__) */

